#version 450
#include <layout.glsl>

float hash11(float x){
    x = fract(x * 0.1031);
    x *= x + 33.33;
    x *= x + x;
    return fract(x);
}

float hash21(vec2 p){
    vec3 p3 = fract(vec3(p.xyx) * 0.1031);
    p3 += dot(p3, p3.yzx + 33.33);
    return fract((p3.x + p3.y) * p3.z);
}

float noise(vec2 p){
    vec2 i = floor(p), f = fract(p);
    vec2 u = f*f*f*(f*(f*6.0 - 15.0) + 10.0);
    float a = hash21(i + vec2(0,0));
    float b = hash21(i + vec2(1,0));
    float c = hash21(i + vec2(0,1));
    float d = hash21(i + vec2(1,1));
    return mix(mix(a,b,u.x), mix(c,d,u.x), u.y);
}

// Barrel distortion for CRT curvature
vec2 barrel(vec2 uvCentered, float k){
    float r2 = dot(uvCentered, uvCentered);
    return uvCentered * (1.0 + k * r2);
}

// RGB phosphor aperture mask (per screen pixel column)
vec3 apertureMask(vec2 uv, vec2 res){
    float col = floor(uv.x * res.x);
    float phase = mod(col, 3.0);
    vec3 m;
    if      (phase < 0.5) m = vec3(1.0, 0.4, 0.4);
    else if (phase < 1.5) m = vec3(0.4, 1.0, 0.4);
    else                  m = vec3(0.4, 0.4, 1.0);
    return mix(vec3(1.0), m, parameters.markStrength);
}

// Source sampling with chromatic aberration and horizontal bleed
vec3 sampleSource(vec2 uv){
    vec2 dx = vec2(parameters.aberration, 0.0);

    vec3 base = texture(sampler2D(inputTexture, inputSampler), uv).rgb;
    vec3 l = texture(sampler2D(inputTexture, inputSampler), uv - vec2(parameters.phosphorBleed,0.0)).rgb;
    vec3 r = texture(sampler2D(inputTexture, inputSampler), uv + vec2(parameters.phosphorBleed,0.0)).rgb;
    vec3 bleed = (base * 0.7 + (l + r) * 0.15);

    vec3 c;
    c.r = texture(sampler2D(inputTexture, inputSampler), uv + dx).r;
    c.g = base.g;
    c.b = texture(sampler2D(inputTexture, inputSampler), uv - dx).b;

    return mix(c, bleed, 0.35);
}

vec2 letterbox(vec2 res, vec2 view) {
    float targetAspect = res.x / res.y;
    float screenAspect = view.x / view.y;
    vec2 uv;

    if (screenAspect > targetAspect) {
        float targetWidth = view.y * targetAspect;
        float offset = (view.x - targetWidth) * 0.5;
        uv = vec2(
            (gl_FragCoord.x - offset) / targetWidth,
            gl_FragCoord.y / view.y
        );
    } else {
        float targetHeight = view.x / targetAspect;
        float offset = (view.y - targetHeight) * 0.5;
        uv = vec2(
            gl_FragCoord.x / view.x,
             (gl_FragCoord.y - offset) / targetHeight
        );
    }

    return uv;
}

void main() {
    vec2 res = vec2(resolution.x, resolution.y);
    vec2 view = vec2(viewport.x, viewport.y);
    vec2 uv = letterbox(res, view);

    vec3 base = texture(sampler2D(inputTexture, inputSampler), uv).rgb;

    if (parameters.blend < 0.00001) {
        outputColor = vec4(base, 1.0);
        return;
    }

    vec2 uvC = uv * 2.0 - 1.0;

    // CRT curvature (barrel)
    uvC = barrel(uvC, parameters.curvature);
    vec2 uvWarp = uvC * 0.5 + 0.5;

    // Soft outer border mask (antialiased to black)
    // edge > 0 when uvWarp lies outside the [0,1]^2 domain
    float edge = max(max(-uvWarp.x, uvWarp.x - 1.0), max(-uvWarp.y, uvWarp.y - 1.0));
    float feather = parameters.borderAAPx / min(res.x, res.y); // convert px to UV units
    float borderMask = 1.0 - smoothstep(0.0, feather, edge); // 1 inside, 0 outside with fade

    // Rolling vertical sweep (bright line moving down)
    float t = time.current;
    float sweepY = fract(t * parameters.sweepSeed);
    float dy = uvWarp.y - sweepY;
    float sweep = exp(-pow(abs(dy) / (parameters.sweepWidth * 0.5), 1.8));

    // Per-scanline jitter, stronger near the sweep
    float lineId = floor(uvWarp.y * res.y);
    float jitter = (hash11(lineId + floor(t * 60.0) * 13.0) - 0.5) * (parameters.jitterPix / res.x);
    vec2 uvJ = uvWarp + vec2(jitter * sweep, 0.0);

    // Clamp sampling to avoid undefined reads while border fades to black
    vec2 uvSample = clamp(uvJ, 0.0, 1.0);

    // Base color from source with aberration/bleed
    vec3 col = sampleSource(uvSample);

    // Scanlines (cosine modulation per row)
    float scan = 0.5 + 0.5 * cos((uvWarp.y * res.y) * 3.14159);
    col *= (1.0 - parameters.scanStrength * (1.0 - scan));

    // Phosphor aperture mask
    col *= apertureMask(uvWarp, res);

    // Additional brightness along the sweep
    col += sweep * 0.07;

    // vignette based on curvature radius
    float r2 = dot(uvC, uvC);
    float vig = 1.0 - parameters.vignette * smoothstep(0.6, 1.0, r2);
    col *= vig;

    // Subtle temporal noise
    float g = noise(gl_FragCoord.xy * 0.75 + t * 37.0) - 0.5;
    col += g * parameters.noiseAmt;

    // Gentle gamma shaping for CRT-like response
    col = pow(max(col, 0.0), vec3(parameters.gamma));

    // Apply outer border AA mask (fade to black)
    col *= borderMask;

    outputColor = vec4(mix(base, clamp(col, 0.0, 1.0), parameters.blend), 1.0);


    //bool h = gl_FragCoord.x < 2 || gl_FragCoord.x > res.x - 2;
    //bool v = gl_FragCoord.y < 2 || gl_FragCoord.y > res.y - 2;
    //if (h || v) {
    //    //outputColor = vec4(1.0, 0.0, 0.0, 1.0);
    //}
}
